# Copyright (c) 2025 CenturyArks Co.,Ltd.
#
# These source code may be used (including the use of modified source code) only when using
# CenturyArks's products or services.
#
# Although we have confirmed that it works in the copyright holder's environment, we cannot guarantee that
# there will be no defects.
#
# If you modify the source code, do not remove the copyright notice.
#
# Copyright holder shall not be liable for any claim, damages or other liability arising
# from or relating to the source code or arising from the use or other dealings in the software.

"""
Sample code demonstrating how to synchronize the SilkyEvCamHD camera using the Metavision HAL Python API.
While the slave is running, it continuously records raw event data.
"""

import argparse
import datetime
import gc
import os
import time
import threading
import queue
import numpy as np
import cv2

from metavision_core.event_io import EventsIterator
from metavision_core.event_io.raw_reader import initiate_device
from metavision_hal import I_TriggerIn
from metavision_sdk_core import ColorPalette, PeriodicFrameGenerationAlgorithm
from metavision_sdk_ui import BaseWindow, EventLoop, MTWindow, UIKeyEvent
from vmbpy import LOG_CONFIG_INFO_CONSOLE_ONLY, Log

from silky_util import BiasConfig, CameraConfig


def print_preamble():
    print('////////////////////////////////////////////////////////')
    print('// "SilkyEvCam [Slave] synced to BothView [Master] /////')
    print('////////////////////////////////////////////////////////')
    print('Press <Esc or Q> to exit. ')
    print(flush=True)


def ranged_type(value_type, min_value, max_value):
    def range_checker(arg: str):
        try:
            f = value_type(arg)
        except ValueError:
            raise argparse.ArgumentTypeError(f'must be a valid {value_type}')
        if f < min_value or f > max_value:
            raise argparse.ArgumentTypeError(f'must be within [{min_value}, {max_value}]')
        return f
    return range_checker


def parse_args():
    parser = argparse.ArgumentParser(description='SilkyEvCamHD slave camera: synchronizes with master and continuously records raw events.',
                                     formatter_class=argparse.ArgumentDefaultsHelpFormatter)

    input_group = parser.add_argument_group('Input', 'Arguments related to input sequence.')
    input_group.add_argument(
        '-se', '--event-camera-serial', dest='event_camera_serial', default='',
        help='Serial number of the event camera. If not specified, the live stream of the first available event camera will be used.')
    input_group.add_argument(
        '-ub', '--using-bias-file', dest='using_bias_file', action='store_true',
        help="Enable the Bias configuration file (. /config/biases.bias). Overrides values in settings.json.")

    image_setting_group = parser.add_argument_group('Image Setting', 'Arguments related to image display.')
    image_setting_group.add_argument(
        '-at', '--accumulation-time-us', dest='accumulation_time_us', default=33333, type=ranged_type(int, 1, 1000000),
        help='Accumulation time for frame generation. (Unit:us)')
    image_setting_group.add_argument(
        '-fr', '--viewer-frame-rate', dest='viewer_frame_rate', default=20, type=ranged_type(int, 1, 30),
        help='Frame rate for viewer display (fps)')

    return parser.parse_args()


def create_overlay(text, frame_shape, color=(0, 0, 255)):
    """Pre-rendered overlay image with given text"""
    overlay = np.zeros_like(frame_shape, dtype=np.uint8)
    cv2.putText(overlay, text, (30, 30),
                cv2.FONT_HERSHEY_COMPLEX_SMALL, 1, (30, 30, 30), 2, cv2.LINE_AA)
    cv2.putText(overlay, text, (30, 30),
                cv2.FONT_HERSHEY_COMPLEX_SMALL, 1, color, 1, cv2.LINE_AA)
    return overlay


def create_dummy_frame(height, width):
    """Dark gray dummy frame with text overlay"""
    frame = np.full((height, width, 3), 70, dtype=np.uint8)  # Dark gray
    return frame


def main():
    print_preamble()

    args = parse_args()
    log = Log.get_instance()
    log.enable(LOG_CONFIG_INFO_CONSOLE_ONLY)

    # HAL device
    device = initiate_device(path=args.event_camera_serial, do_time_shifting=False)

    # Recording file
    file_name = 'slave_recording_' + datetime.datetime.now().strftime('%y%m%d_%H%M%S_%f')[:-3]
    file_path = os.path.join('.', 'video', file_name + '.raw')
    device.get_i_events_stream().log_raw_data(file_path)

    # Set device to slave mode
    if device.get_i_camera_synchronization():
        device.get_i_camera_synchronization().set_mode_slave()
        log.info('Set mode slave successful. Start master camera to launch streaming.')

    log.info('Recording to {} started with streaming and will stop upon viewer exit.'.format(file_path))

    # Events iterator on Camera
    mv_iterator = EventsIterator.from_device(
        device=device, delta_t=2e4, relative_timestamps=False)

    # Get camera resolution
    height, width = mv_iterator.get_size()

    # Determine if camera is HD or VGA
    is_hd = (width >= 1280 and height >= 720)  # True: HD, False: VGA
    cam_type = "HD" if is_hd else "VGA"
    log.info(f"Camera detected: {cam_type} ({width}x{height})")

    # Camera / Bias settings
    if is_hd:
        cc = CameraConfig(device)
        cc.set_from_file(file_name='./config/settings.json', disp_flag=False)
        if args.using_bias_file:
            biases = mv_iterator.reader.device.get_i_ll_biases()
            bc = BiasConfig(biases)
            bc.set_from_file(file_name='./config/biases.bias')

    # InputTrigger
    i_trigger_in = device.get_i_trigger_in()
    i_trigger_in.enable(I_TriggerIn.Channel.MAIN)

    # Event Frame Generator
    event_frame_gen = PeriodicFrameGenerationAlgorithm(
        sensor_width=width, sensor_height=height,
        fps=args.viewer_frame_rate,
        accumulation_time_us=args.accumulation_time_us,
        palette=ColorPalette.Dark
    )

    # --- Pre-rendered overlays ---
    dummy_image = create_dummy_frame(height, width)
    dummy_overlay = create_overlay("Waiting for sync signal...", dummy_image)
    rec_overlay = create_overlay("[REC]", dummy_image, color=(0, 0, 255))
    frame_to_show = cv2.addWeighted(dummy_image, 1.0, dummy_overlay, 1.0, 0)

    # GUI
    title_plus = args.event_camera_serial
    title = f"SilkyEvCam {cam_type} {title_plus} [Slave] synced to BothView [Master]"
    with MTWindow(title=title, width=width, height=height,
                  mode=BaseWindow.RenderMode.BGR) as window:

        def keyboard_cb(key, scancode, action, mods):
            if key == UIKeyEvent.KEY_ESCAPE or key == UIKeyEvent.KEY_Q:
                window.set_close_flag()
        window.set_keyboard_callback(keyboard_cb)

        def on_cd_frame_cb(ts, cd_frame):
            """Display event frame immediately once generated with overlay"""
            nonlocal frame_to_show
            frame_to_show = cv2.addWeighted(cd_frame, 1.0, rec_overlay, 1.0, 0)
        event_frame_gen.set_output_callback(on_cd_frame_cb)

        last_gc_time = time.time()
        local_ext_trigger_count = 0

        # --- Queue for receiving events ---
        event_queue = queue.Queue(maxsize=1)

        # --- mv_iterator thread ---
        def poll_events():
            nonlocal local_ext_trigger_count
            for evs in mv_iterator:  # Blocking wait
                triggers = mv_iterator.reader.get_ext_trigger_events()
                if triggers.size > 0:
                    local_ext_trigger_count += len(triggers)
                    mv_iterator.reader.clear_ext_trigger_events()

                # Keep only the latest frame in the queue
                while event_queue.full():
                    try:
                        event_queue.get_nowait()  # discard the old frame
                    except queue.Empty:
                        break
                event_queue.put_nowait(evs)

        threading.Thread(target=poll_events, daemon=True).start()

        # --- Main loop ---
        window.show_async(cv2.addWeighted(dummy_image, 1.0, dummy_overlay, 1.0, 0))

        while not window.should_close():
            EventLoop.poll_and_dispatch()

            try:
                evs = event_queue.get(timeout=0.001)
                event_frame_gen.process_events(evs)
                del evs
            except queue.Empty:
                # CPU spin suppression with 1 ms sleep
                time.sleep(0.001)

            # Show latest frame
            window.show_async(frame_to_show)

            # Garbage collection
            now = time.time()
            if now - last_gc_time >= 5.0:
                gc.collect()
                last_gc_time = now

        log.info('Event-raw Recording Stopped.')
        log.info('Event Actual Triggers: ' + str(local_ext_trigger_count))
        device.get_i_events_stream().stop_log_raw_data()


if __name__ == "__main__":
    main()
