# Copyright (c) 2025 CenturyArks Co.,Ltd.
#
# These source code may be used (including the use of modified source code) only when using
# CenturyArks's products or services.
#
# Although we have confirmed that it works in the copyright holder's environment, we cannot guarantee that
# there will be no defects.
#
# If you modify the source code, do not remove the copyright notice.
#
# Copyright holder shall not be liable for any claim, damages or other liability arising
# from or relating to the source code or arising from the use or other dealings in the software.

"""
Silky Camera(event-base) View Started.
"""

import gc
import os
import time
from multiprocessing.synchronize import Lock as LockType

import numpy
from metavision_core.event_io import EventsIterator
from metavision_core.event_io.raw_reader import initiate_device
from metavision_hal import I_TriggerIn
from metavision_sdk_core import ColorPalette, PeriodicFrameGenerationAlgorithm
from metavision_sdk_ui import EventLoop
from vmbpy import LOG_CONFIG_INFO_CONSOLE_ONLY, Log

import global_define as g_define
from shared_context import get_ctx, init_ctx
from shared_params import SharedParams
from silky_util import BiasConfig, CameraConfig


def main(shm_name: str, shm_lock: LockType, shared_params: SharedParams):
    """ Main """
    args = g_define.get_args()

    log = Log.get_instance()
    log.enable(LOG_CONFIG_INFO_CONSOLE_ONLY)

    # Redefine “SharedContext” in the child process
    init_ctx(name=shm_name, create=False,
             lock=shm_lock, shared_params=shared_params)
    ctx = get_ctx()

    # first we create a HAL device
    try:
        device = initiate_device(path=args.event_camera_serial, do_time_shifting=False)
    except (OSError):
        log.error('[Error] Event camera not found. Check the USB connection.')
        ctx.params.event_camera_err_flag.set()

    ctx.params.event_camera_open_proc.set()
    if ctx.params.event_camera_err_flag.is_set():
        return

    # Wait for the end of the 'frame camera' opening
    ctx.params.frame_camera_open_proc.wait()
    if ctx.params.frame_camera_open_proc.is_set():
        if ctx.params.frame_camera_err_flag.is_set():
            return  # unavoidably terminated

    # Set the multiple device synchronization mode to master/slave.
    # Single mode if unspecified.
    if device.get_i_camera_synchronization():
        if args.cam_mode == 'master':
            device.get_i_camera_synchronization().set_mode_master()
            log.info('[Event-Master] Set mode master successful. Make sure to start slave camera first')
        elif args.cam_mode == 'slave':
            device.get_i_camera_synchronization().set_mode_slave()
            log.info('[Event-Slave] Set mode slave successful. Start master camera to launch streaming ')

    # Events iterator on Camera
    frame_rate = args.viewer_frame_rate  # fps
    frame_period_us = 2e4  # Adjust delta_t to prevent event display from lagging behind frame display
    mv_iterator = EventsIterator.from_device(
        device=device, delta_t=frame_period_us, relative_timestamps=False)
    height, width = mv_iterator.get_size()  # Camera Geometry

    # Camera Setting
    cc = CameraConfig(device)
    cc.set_from_file(file_name='./config/settings.json', disp_flag=False)

    # Biases Setting
    if args.using_bias_file:
        biases = mv_iterator.reader.device.get_i_ll_biases()
        bc = BiasConfig(biases)
        bc.set_from_file(file_name='./config/biases.bias')

    erc = mv_iterator.reader.device.get_i_erc_module()
    erc_mevs = erc.get_cd_event_rate() / 1e6  # Ev/s -> MEv/s
    log.info('--> [EventCamera] ERC: \'%s\' Mev/s ' % str(erc_mevs))

    # Event Frame Generator
    event_frame_gen = PeriodicFrameGenerationAlgorithm(sensor_width=width, sensor_height=height, fps=frame_rate,
                                                       accumulation_time_us=args.accumulation_time_us, palette=ColorPalette.Dark)

    def on_cd_frame_cb(ts, cd_frame):
        if args.disable_reverse:
            event_buf = cd_frame
        else:
            # Reverse Mode
            event_buf = numpy.ascontiguousarray(cd_frame[:, ::-1])

        ctx.image.update_image(event_buf)
        del event_buf

    event_frame_gen.set_output_callback(on_cd_frame_cb)

    init_recording = False
    last_gc_time = time.time()

    local_ext_trigger_count = 0
    for evs in mv_iterator:
        EventLoop.poll_and_dispatch()  # Dispatch system events to the window

        if len(evs) == 0:
            continue

        # Process events
        event_frame_gen.process_events(evs)
        del evs

        # Perform garbage collection every 5 seconds based on real elapsed time
        now = time.time()
        if now - last_gc_time >= 5.0:
            gc.collect()
            last_gc_time = now

        # Catch main end event
        if ctx.params.is_alive.value is False:
            break

        # Catch main recording event
        if ctx.params.is_evt_recording.value is True:
            if not init_recording:

                # InputTrigger Setting
                i_trigger_in = device.get_i_trigger_in()
                i_trigger_in.enable(I_TriggerIn.Channel.MAIN)

                log_path = ctx.params.recording_name.get() + '.raw'

                log_path = os.path.join(args.recording_dir_path, log_path)
                log.info('Event-Raw Recording to {} Start...'.format(log_path))
                device.get_i_events_stream().log_raw_data(log_path)

            init_recording = True

            triggers = mv_iterator.reader.get_ext_trigger_events()
            if len(triggers) > 0:
                local_ext_trigger_count += len(triggers)
            mv_iterator.reader.clear_ext_trigger_events()

        else:
            if init_recording:
                # InputTrigger Setting
                i_trigger_in = device.get_i_trigger_in()
                i_trigger_in.disable(I_TriggerIn.Channel.MAIN)

                log.info('Event-Raw Recording Stop...')
                device.get_i_events_stream().stop_log_raw_data()
                ctx.params.ext_trigger_count.value = local_ext_trigger_count
                local_ext_trigger_count = 0

            init_recording = False

    del device
    gc.collect()
    log.info('Thread \'Event-Camera\' terminated.')


if __name__ == '__main__':

    log = Log.get_instance()
    log.enable(LOG_CONFIG_INFO_CONSOLE_ONLY)
    log.error('[Error] Direct calls are not used.')

    # Memo:
    # For stand-alone debugging
    '''
    shm_name = 'my_shm'
    shm_lock = Lock()
    manager = Manager()
    shared_params = SharedParams(manager)
    init_ctx(name=shm_name, create=True, lock=shm_lock, shared_params=shared_params)

    main(shm_name, shm_lock)
    '''
