# Copyright (c) 2025 CenturyArks Co.,Ltd.
#
# These source code may be used (including the use of modified source code) only when using
# CenturyArks's products or services.
#
# Although we have confirmed that it works in the copyright holder's environment, we cannot guarantee that
# there will be no defects.
#
# If you modify the source code, do not remove the copyright notice.
#
# Copyright holder shall not be liable for any claim, damages or other liability arising
# from or relating to the source code or arising from the use or other dealings in the software.

"""
Alvium Camera(global shutter) View Started.
"""

import configparser
import gc
import math
import multiprocessing
import os
import queue
import re
import subprocess
import sys
import threading
import time
from datetime import datetime
from multiprocessing.synchronize import Lock as LockType
from typing import Optional

import cv2
import ffmpeg
import numpy
from metavision_sdk_ui import BaseWindow, EventLoop, MTWindow, UIKeyEvent
from vmbpy import (LOG_CONFIG_INFO_CONSOLE_ONLY, Camera, CameraEvent, Frame,
                   FrameStatus, Log, PersistType, PixelFormat, Stream,
                   VmbCameraError, VmbFeatureError, VmbSystem)

import bothview_event as silky_cam
import global_calc as g_calc
import global_define as g_define
import global_value as g_value
from shared_context import get_ctx, init_ctx
from shared_params import SharedParams

# 'config.ini' file value getting
config = configparser.ConfigParser()
config.read('./config/config.ini', encoding="utf-8")
conf_camera01 = 'camera01_spec'
conf_camera02 = 'camera02_spec'
common = 'common'
user_setting = 'user_setting'
FRAME_RESOLUTION_W = int(config.get(conf_camera01, 'resolution_w'))
FRAME_RESOLUTION_H = int(config.get(conf_camera01, 'resolution_h'))
EVENT_RESOLUTION_W = int(config.get(conf_camera02, 'resolution_w'))
EVENT_RESOLUTION_H = int(config.get(conf_camera02, 'resolution_h'))
FRAME_PIXCEL_W = float(config.get(conf_camera01, 'pixcel_w'))
FRAME_PIXCEL_H = float(config.get(conf_camera01, 'pixcel_h'))
EVENT_PIXCEL_W = float(config.get(conf_camera02, 'pixcel_w'))
EVENT_PIXCEL_H = float(config.get(conf_camera02, 'pixcel_h'))
FRAME_QUEUE_SIZE = int(config.get(common, 'frame_queue_size'))
ADJUST_VIEW_W = int(config.get(user_setting, 'adjust_view_w'))
ADJUST_VIEW_H = int(config.get(user_setting, 'adjust_view_h'))

# Multi-view mode: Explanatory image
desc_img1 = cv2.imread('images/description.jpg')
desc_img2 = cv2.imread('images/multi_frame_discription.jpg')
desc_list = [desc_img1, desc_img2]


def print_preamble():
    print('///////////////////////////////////////////////')
    print('// SilkyEvCam - BothView                  /////')
    print('///////////////////////////////////////////////')
    print('Press <Esc or Q> to exit. ')
    print('Press <V> to Change View Mode.')
    print('Press <R> to Recording(2video mode) Start or End. ')
    print('Press <M> to Recording(multi-view mode) Start or End. ')
    print(flush=True)


def abort(reason: str, return_code: int = 1):
    print(reason + '\n')
    sys.exit(return_code)


def concat_tile(im_list_2d):
    return cv2.vconcat([cv2.hconcat(im_list_h) for im_list_h in im_list_2d])


def change_from_frame_to_opencv(frame: Frame, id_no: int = -1) -> numpy.ndarray:
    cv_frame = frame.as_opencv_image()

    if id_no != -1:
        cv2.putText(cv_frame, str(id_no), org=(g_value.write_frame_id_x, g_value.write_frame_id_y),
                    fontScale=2, color=(0, 0, 255), thickness=1, fontFace=cv2.FONT_HERSHEY_COMPLEX_SMALL, lineType=cv2.LINE_AA)

    return cv_frame


def try_put_frame(q: queue.Queue, cam: Camera, cv_frame: Optional[numpy.ndarray]):
    # Add frame information of the target camera (id) to the queue.
    try:
        q.put_nowait((cam.get_id(), cv_frame))

    except queue.Full:
        pass


def ffmpeg_final_stats_log_worker(stderr_pipe, logfile_path):

    # Save only the final statistics log
    last_stats_line = None
    for line in iter(stderr_pipe.readline, b''):
        decoded = line.decode('utf-8').strip()

        # The final statistics log of ffmpeg follows the format: "frame=... fps=... Lsize=... time=... bitrate=... speed=...".
        if decoded.startswith("frame=") and "time=" in decoded and "bitrate=" in decoded and "speed=" in decoded:
            last_stats_line = decoded

    if last_stats_line:
        with open(logfile_path, 'w', encoding='utf-8') as f:
            f.write("[ffmpeg log] " + last_stats_line + '\n')


def createMp4Process(img_w: int, img_h: int) -> subprocess.Popen:
    # Generate process during recording
    ctx = get_ctx()  # Access "SharedContext" set in parent process
    log = Log.get_instance()
    args = g_define.get_args()

    # Constants
    _CODEC_DICT = {'GPU': 'h264_nvenc', 'CPU': 'h264'}  # No used 'AMF':'h264_amf', 'QSV':'h264_qsv'

    v_codec = _CODEC_DICT['GPU'] if args.frame_using_gpu else _CODEC_DICT['CPU']

    file_path = ctx.params.recording_name.get() + '.mp4'
    file_path = os.path.join(args.recording_dir_path, file_path)

    if args.frame_using_gpu:
        # Memo:
        # When using "passthrough" with GPU encoding, you cannot control the bitrate.
        # Using "preset='llhp'" as an alternative option (speed priority).
        ffmpeg_process = (
            ffmpeg
            .input('pipe:', format='rawvideo', pix_fmt='bgr24',
                   s='{}x{}'.format(img_w, img_h), use_wallclock_as_timestamps=1, fflags='+genpts')
            .crop(g_value.img_trim_offset_x, g_value.img_trim_offset_y, g_value.img_trim_width, g_value.img_trim_height)
            .output(file_path, pix_fmt='yuv420p', vcodec=v_codec, vsync='passthrough',
                    preset='llhp', bf='0', loglevel="info", stats=None)  # loglevel-default:info
            .run_async(pipe_stdin=True, pipe_stderr=True, overwrite_output=True)  # Display log on Text
        )

    else:
        # Memo:
        # To minimize missing frames, "vsync=passthrough" is used.
        # Normally set "vsync=vfr".
        ffmpeg_process = (
            ffmpeg
            .input('pipe:', format='rawvideo', pix_fmt='bgr24',
                   s='{}x{}'.format(img_w, img_h), use_wallclock_as_timestamps=1, fflags='+genpts')
            .crop(g_value.img_trim_offset_x, g_value.img_trim_offset_y, g_value.img_trim_width, g_value.img_trim_height)
            .output(file_path, pix_fmt='yuv420p', vcodec=v_codec, vsync='passthrough',
                    preset='ultrafast', bf='0', tune='zerolatency', loglevel="info", stats=None)  # loglevel-default:info
            .run_async(pipe_stdin=True, pipe_stderr=True, overwrite_output=True)  # Display log on Text
        )

    # Log text output processing is executed in a thread
    log_path = ctx.params.recording_name.get() + '_ffmpeg-final-stats.txt'
    log_path = os.path.join(args.recording_dir_path, log_path)
    log_thread = threading.Thread(
        target=ffmpeg_final_stats_log_worker,
        args=(ffmpeg_process.stderr, log_path),
        daemon=True
    )
    log_thread.start()

    log.info('Frame-MP4 Recording to {} Start...'.format(file_path))
    return ffmpeg_process


def destroyMp4Process(ffmpeg_process: subprocess.Popen):
    # Destroys the process being recorded
    log = Log.get_instance()

    if ffmpeg_process is not None:
        log.info('Frame-MP4 Recording Stop...')
        ffmpeg_process.stdin.close()
        ffmpeg_process.wait()
        ffmpeg_process = None


# Thread Objects
class FrameProducer(threading.Thread):
    def __init__(self, cam: Camera, frame_queue: queue.Queue, event_queue: queue.Queue):
        threading.Thread.__init__(self)

        self.log = Log.get_instance()
        self.cam = cam
        self.frame_queue = frame_queue
        self.event_queue = event_queue
        self.killswitch = threading.Event()
        self.stop_polling_event = threading.Event()

        # Instance Variables for Recordings
        self.init_recording = False
        self.ffmpeg_process = None

        # Counter to track the number of processed frames
        self.frame_count = 0
        self.viewer_step = 1
        self.cam_setting_get_flag = False

        # For log text output

        self.log_queue = queue.Queue()
        self.log_thread = threading.Thread(target=self.frame_drop_log_worker, daemon=True)
        self.log_thread.start()

    def frame_drop_log_worker(self):
        # Memo:
        # - If no target logs are output, the text file will not be created.
        # - Frame IDs start at 0

        last_fsync = time.time()
        while True:
            log_path, log_text = self.log_queue.get()
            if log_path is None:
                break
            with open(log_path, 'a', encoding='utf-8') as f:
                f.write(log_text)
                f.flush()
                now = time.time()
                if now - last_fsync >= 1:  # Every second
                    os.fsync(f.fileno())
                    last_fsync = now

            self.log_queue.task_done()

    def __call__(self, cam: Camera, stream: Stream, frame: Frame):

        args = g_define.get_args()
        ctx = get_ctx()  # Access "SharedContext" set in parent process

        # This method is executed within VmbC context. All incoming frames
        # are reused for later frame acquisition. If a frame shall be queued, the
        # frame must be copied and the copy must be sent, otherwise the acquired
        # frame will be overridden as soon as the frame is reused.
        if frame.get_status() == FrameStatus.Complete:

            # Display frame number
            # print('{} acquired {}'.format(cam, frame), flush=True)

            # All frames will either be recorded in this format, or transformed to it before being displayed
            opencv_display_format = PixelFormat.Bgr8

            # Convert frame if it is not already the correct format
            if frame.get_pixel_format() == opencv_display_format:
                display = frame
            else:
                # This creates a copy of the frame. The original `frame` object can be requeued
                # safely while `display` is used
                display = frame.convert_pixel_format(opencv_display_format)

            frame_id = frame.get_id() if args.write_frame_id else -1
            cv_frame = change_from_frame_to_opencv(display, frame_id)
            del display

            # Memo:
            # Thin out to the specified frame rate value.
            if self.frame_count % self.viewer_step == 0:
                if not self.frame_queue.full():
                    try_put_frame(self.frame_queue, cam, cv_frame)

            if g_value.is_recording is True:
                try:
                    g_value.FFMPEG_PROCESS.stdin.write(cv_frame)
                    g_value.RECORDING_FRAME_CNT += 1
                except Exception as e:
                    # Select 'GPU' and follow up if GPU is not available.
                    self.log.error('ffmpeg error:' + str(e))
                    g_value.is_recording = False
                    ctx.params.is_evt_recording.value = False

        else:
            # Memo:
            # If this point is reached during recording,
            # the mp4 file will have no frames.
            # This also happens when outputting a file
            # with "FFMPEG_PROCESS". (The log output will show "drop=n").
            #
            # The cause depends mainly on the number of times the FPS value is processed
            # and computer specs, so adjust accordingly.

            if g_value.is_recording is True:
                log_text = f'Missing frame... frame info: {frame}\n'
                log_path = os.path.join(args.recording_dir_path, ctx.params.recording_name.get() + '_lost-cam-frame.txt')
                self.log_queue.put((log_path, log_text))
                g_value.RECORDING_FRAME_CNT += 1

        self.frame_count += 1
        cam.queue_frame(frame)

    def stop(self):
        self.killswitch.set()

    def setup_camera(self):

        args = g_define.get_args()

        # 1. Initialize 'ROI-related' settings
        self.cam.BinningHorizontal.set(1)
        self.cam.BinningVertical.set(1)
        width_max = self.cam.WidthMax.get()
        height_max = self.cam.HeightMax.get()
        self.cam.OffsetX.set(0)
        self.cam.OffsetY.set(0)
        self.cam.Width.set(width_max)
        self.cam.Height.set(height_max)

        # 2.Sensor size Calculation START
        frame_sensor_w, frame_sensor_h = g_calc.get_cencer_size((FRAME_RESOLUTION_W, FRAME_RESOLUTION_H),
                                                                (FRAME_PIXCEL_W, FRAME_PIXCEL_H))

        event_sensor_w, event_sensor_h = g_calc.get_cencer_size((EVENT_RESOLUTION_W, EVENT_RESOLUTION_H),
                                                                (EVENT_PIXCEL_W, EVENT_PIXCEL_H))

        # 3.Image Resolution size Calculation START
        frame_trim_w, frame_trim_h = g_calc.get_trim_pixel_size((FRAME_RESOLUTION_W, FRAME_RESOLUTION_H),
                                                                (EVENT_RESOLUTION_W, EVENT_RESOLUTION_H),
                                                                (frame_sensor_w, frame_sensor_h),
                                                                (event_sensor_w, event_sensor_h))

        # 4.Frame size calculation for ROI specification
        frame_roi_w = FRAME_RESOLUTION_W - (frame_trim_w * 2)
        frame_roi_h = FRAME_RESOLUTION_H - (frame_trim_h * 2)

        # 5.Adjust ROI to meet frame camera specifications
        ((frame_roi_w, frame_roi_h), (frame_trim_w, frame_trim_h)) = g_calc.get_adjusted_roi((frame_roi_w, frame_roi_h),
                                                                                             (frame_trim_w, frame_trim_h))

        # 6.Obtain ROI information (for image cropping)
        # * Because the Frame ROI Interval is large in 4px increments, Crop the image itself.

        # Frame Position Adjustment
        adjust_w, adjust_h = ADJUST_VIEW_W, ADJUST_VIEW_H
        if args.disable_adjust:
            adjust_w, adjust_h = 0, 0

        g_value.img_trim_width = int(frame_roi_w)
        g_value.img_trim_height = int(frame_roi_h)
        g_value.img_trim_offset_x = int(frame_trim_w - g_calc.get_adjusted_offset(adjust_w))
        g_value.img_trim_offset_y = int(frame_trim_h + g_calc.get_adjusted_offset(adjust_h))
        g_value.write_frame_id_x = 40 + g_value.img_trim_offset_x
        g_value.write_frame_id_y = 60 + g_value.img_trim_offset_y

        # 7.Initialize output trigger settings
        self.cam.LineSelector.set('Line0')
        self.cam.LineMode.set('Output')
        self.cam.LineSource.set('Off')
        self.cam.LineInverter.set(True)

        # 8.Enable Acquisition FrameRate Control
        try:
            if args.eaf_mode:
                self.log.info('--> [setup_camera] AcquisitionFrameRateEnable (value: %sfps)' % args.acquisition_frame)
                self.cam.AcquisitionFrameRateEnable.set(1)  # Mode On
                self.cam.AcquisitionFrameRate.set(args.acquisition_frame)  # FPS

        except VmbFeatureError:
            self.log.warning('\'AcquisitionFrameRate\' is not set.')
            pass

        # 9.Calculate the number of steps to determine
        # the frame rate of the viewer separately from the frame rate of the camera.
        # (Used within the ‘__call__’ function)
        try:
            acquisition_frame_rate = int(self.cam.AcquisitionFrameRate.get())
            downsample_target = args.viewer_frame_rate
            self.viewer_step = max(1, acquisition_frame_rate // downsample_target)

        except (AttributeError, VmbFeatureError):
            pass

        # 10.Multiple device synchronization mode
        # Set the multiple device synchronization mode to master/slave.
        # Single mode if unspecified.
        try:
            if args.cam_mode == 'master':
                self.cam.LineSelector.set('Line1')
                self.cam.LineMode.set('Output')
                self.cam.LineSource.set('ExposureActive')
                self.cam.LineInverter.set(False)
                self.log.info('[Frame-Master] Set mode master successful. Make sure to start slave camera first')

                # Memo:
                # Immediately after the camera is activated, the status is set to “True”.
                # Therefore, “True” is set as recording stopped.
                self.cam.LineSelector.set('Line2')
                self.cam.LineMode.set('Output')
                self.cam.LineInverter.set(True)  # False=ON, True=OFF

            elif args.cam_mode == 'slave':
                # Memo:
                # Acquisition Frame Rate is used only if Frame Start Trigger Mode is set to 'Off'
                if self.cam.AcquisitionFrameRateEnable.get() is True:
                    self.cam.AcquisitionFrameRateEnable.set(False)

                self.cam.TriggerSource.set('Line1')
                self.cam.TriggerSelector.set('FrameStart')
                self.cam.TriggerMode.set('On')
                self.log.info('[Frame-Slave] Set mode slave successful. Start master camera to launch streaming ')

                self.cam.LineSelector.set('Line2')
                self.cam.LineMode.set('Input')

        except (AttributeError, VmbFeatureError):
            pass

        # Memo:
        # Try to enable automatic exposure time setting
        '''
        try:
            self.cam.ExposureAuto.set('Once')

        #except (AttributeError, VmbFeatureError):
            self.log.info('Camera {}: Failed to set Feature \'ExposureAuto\'.'.format(
                        self.cam.get_id()))
        '''

        # Memo:
        # Try the basic color settings. If none are available, it will default to monochrome.
        # (Currently, this is the default value in "settings.xml")
        '''
        cv_fmts = intersect_pixel_formats(self.cam.get_pixel_formats(), OPENCV_PIXEL_FORMATS)
        color_fmts = intersect_pixel_formats(cv_fmts, COLOR_PIXEL_FORMATS)

        if color_fmts:
            self.cam.set_pixel_format(color_fmts[0])

        else:
            mono_fmts = intersect_pixel_formats(cv_fmts, MONO_PIXEL_FORMATS)

            if mono_fmts:
                self.cam.set_pixel_format(mono_fmts[0])

            else:
                abort('Camera does not support a OpenCV compatible format natively. Abort.')
        '''

        # Log primary Frame Camera configuration values
        if args.cam_mode == 'slave':
            self.log.info("--> [FrameCamera] Acquisition Frame Rate: follows the Master's frame rate setting")
        else:
            self.log.info("--> [FrameCamera] Acquisition Frame Rate: '%s' fps" %
                          round(self.cam.AcquisitionFrameRate.get(), 2))

        self.log.info('--> [FrameCamera] Viewer Frame Rate: \'%s\' fps ' % int(args.viewer_frame_rate))
        self.log.info('--> [FrameCamera] Pixcel Format: \'%s\' ' % self.cam.get_pixel_format())
        self.log.info('--> [FrameCamera] Exposure Time: \'%s\' us ' % self.cam.ExposureTime.get())
        self.log.info('--> [FrameCamera] Gain: \'%s\' db ' % self.cam.Gain.get())
        self.log.info('--> [FrameCamera] Sensor Bit Depth: \'%s\' ' % self.cam.get_feature_by_name("SensorBitDepth").get())
        self.log.info('--> [FrameCamera] Device Link Throughput Limit: \'%s\' Bytes/s '
                      % self.cam.get_feature_by_name("DeviceLinkThroughputLimit").get())
        self.log.info('--> [FrameCamera] Device Temperature: \'%s\' °C ' % self.cam.get_feature_by_name("DeviceTemperature").get())

    # Reads the camera's configuration XML file.
    def load_setting_camera(self):

        self.log.info('--> [load_setting_camera] Frame Camera has been opened ({}[{}])'.format(self.cam.get_serial(), self.cam.get_id()))

        # Reset the camera settings to default
        user_set_selector = self.cam.get_feature_by_name("UserSetSelector")
        user_set_selector.set("Default")
        user_set_load = self.cam.get_feature_by_name("UserSetLoad")
        user_set_load.run()

        # Load camera settings from file.
        settings_file = './config/settings.xml'
        self.cam.load_settings(settings_file, PersistType.All)

        self.log.info('--> [load_setting_camera] Feature values have been loaded from given file \'%s\'' % settings_file)

    def monitorRecTrigger(self, interval=0.5):
        # Controls the recording process on the slave side in the synchronization process.
        while not self.stop_polling_event.is_set():
            try:
                self.cam.LineSelector.set('Line2')
                line_status = self.cam.LineStatus.get()
            except AttributeError:
                # Skip attribute errors that occur when the camera is stopped at the end of processing.
                continue

            event = "SYNC_RECORDING_ON" if line_status is False else "SYNC_RECORDING_OFF"

            # Overwrite if full (since maxsize=1, this ensures only the latest remains)
            if self.event_queue.full():
                try:
                    self.event_queue.get_nowait()  # discard old event
                except queue.Empty:
                    pass
            self.event_queue.put_nowait(event)

            # Specify polling interval here (seconds)
            time.sleep(interval)

    def run(self):
        WAIT_TIME_SHORT = 1  # seconds: short wait for stabilization

        args = g_define.get_args()
        self.log.info('Thread \'FrameProducer({}[{}])\' started.'.format(self.cam.get_serial(), self.cam.get_id()))

        try:

            # Memo:
            # As a workaround to stabilize streaming, we reopen the camera after
            # saving UserSet1, following the frame camera manufacturer's recommendation.
            with self.cam:

                # Memo:
                # Since streaming interruptions may be caused by loading the XML
                # configuration file, this specification allows suppressing the
                # loading process when --skip-xml-load is specified.
                if not args.skip_xml_load:
                    # Load XML (Existing Process)
                    self.load_setting_camera()
                else:
                    # Memo: XML load is skipped
                    self.log.info("--> [load_setting_camera] XML load skipped as --skip-xml-load was specified")

                # Save UserSet1 & Set as Default
                try:
                    user_set_selector = self.cam.get_feature_by_name("UserSetSelector")
                    user_set_selector.set("UserSet1")

                    if not args.skip_xml_load:
                        user_set_save = self.cam.get_feature_by_name("UserSetSave")
                        user_set_save.run()

                    user_set_default = self.cam.get_feature_by_name("UserSetDefault")
                    user_set_default.set("UserSet1")

                    action = (
                        "saved and set as the default"
                        if not args.skip_xml_load
                        else "set as the default (skipped saving)"
                    )

                    self.log.info(f"--> [run] UserSet1 has been {action}")

                except Exception as e:
                    self.log.error(f"Failed to configure UserSet1: {e}")

            # The camera (self.cam) is closed here
            # UserSet1 will have been saved on the camera

            time.sleep(WAIT_TIME_SHORT)  # Wait

            # Second camera opening (with UserSet1 applied)
            with self.cam:
                self.setup_camera()
                time.sleep(WAIT_TIME_SHORT)  # Wait

                if args.cam_mode == 'slave':
                    thd_polling = threading.Thread(target=self.monitorRecTrigger)
                    thd_polling.start()

                try:
                    self.log.info('--> [run] Waiting for Start of Frame Streaming...')
                    time.sleep(WAIT_TIME_SHORT)  # Wait
                    self.cam.start_streaming(self)
                    self.killswitch.wait()

                finally:
                    self.cam.stop_streaming()

                    # Reset UserSetDefault to default
                    try:
                        user_set_default = self.cam.get_feature_by_name("UserSetDefault")
                        user_set_default.set("Default")

                    except Exception as e:
                        self.log.error(f"Failed to Reset UserSetDefault to default: {e}")

                    if args.cam_mode == 'slave':
                        self.stop_polling_event.set()
                        thd_polling.join()

        except VmbCameraError:
            pass

        finally:
            try_put_frame(self.frame_queue, self.cam, None)

        self.log.info('Thread \'FrameProducer({}[{}])\' terminated.'.format(self.cam.get_serial(), self.cam.get_id()))


class FrameConsumer(threading.Thread):
    # Performs main "video compositing" processing

    # Constants
    _VIEW_DICT = {'multi': 0, 'both': 1, 'event': 2, 'frame': 3}
    _IMAGE_CAPTION = 'SilkyEvCam - BothView'

    # Memo:
    # Default arguments are set to None to allow calling this method
    # without parameters, anticipating usage from external code.(def execRecResultAnalysis)
    def __init__(self, cam: Camera = None, frame_queue: queue.Queue = None, event_queue: queue.Queue = None):
        threading.Thread.__init__(self)

        self.log = Log.get_instance()
        self.cam = cam
        self.frame_queue = frame_queue
        self.event_queue = event_queue
        self.window = None

        # Variables for propagation to 'callback functions'
        self.quit_mode = 0
        self.view_mode = 0
        self.record_mode = 0
        self.as_seen_record_mode = 0

        # For multi-view mode only: Explanatory image
        self.init_multi_view = False  # Multi-mode initial display
        self.desc_img1 = cv2.imread('images/description.jpg')
        self.desc_img2 = cv2.imread('images/multi_frame_discription.jpg')
        self.desc_list = [self.desc_img1, self.desc_img2]

        # Overlay for status indication
        self.overlay = None
        self.overlay_mode = None

        # Instance Variables for Recordings
        self.init_recording = False
        self.ffmpeg_process = None

        self.recording_on_processed = False
        self.recording_off_processed = False
        self.skip_processing = True  # Initially set to skip “SYNC_REDORDING_OFF” processing

    # Stopping the camera
    def closeCamera(self):
        WAIT_TIME_SHORT = 1  # seconds: short wait for stabilization
        self.log.info('Window close process...')

        ctx = get_ctx()  # Access "SharedContext" set in parent process
        ctx.params.is_alive.value = False

        if self.window:
            self.window.set_close_flag()
            time.sleep(WAIT_TIME_SHORT)  # Wait
            self.window.destroy()

    def execRecResultAnalysis(self, file_path: str, output_result_trigger: bool = True):
        ctx = get_ctx()  # Access "SharedContext" set in parent process
        g_value.is_rec_result_proc = True

        self.log.info('Start analyzing file...')
        f_total = self.getResultFrameNum(file_path + '.mp4')

        log_list = []
        log_list = ['Event Actual Triggers: ' + str(ctx.params.ext_trigger_count.value)]

        if output_result_trigger:
            e_total = self.getResultEventNum(file_path + '.raw')
            log_list += ['Event Result Triggers: ' + str(e_total)]

        log_list += ['Frame Actual Count: ' + str(g_value.RECORDING_FRAME_CNT),
                     'Frame Result Count: ' + str(f_total),
                     'Frame Diff   Count: ' + str(g_value.RECORDING_FRAME_CNT - f_total)]

        result_path = file_path + '_result.txt'
        with open(result_path, mode='w') as f:
            f.writelines([x + '\n' for x in log_list])
            f.close()
        self.log.info('Finish analyzing file... The text of the recording results is output to \'{}\''.format(result_path))
        ctx.params.ext_trigger_count.value = 0
        g_value.is_rec_result_proc = False

    def getResultFrameNum(self, file_path: str):

        cap = cv2.VideoCapture(file_path)
        total = int(cap.get(cv2.CAP_PROP_FRAME_COUNT))

        return total

    def getResultEventNum(self, file_path: str):

        cmd = 'metavision_file_info -i '
        regix = r'External triggers(\s+)([0-9]*)(\s+)'
        exec = cmd + '"' + file_path + '"'
        result = 0

        try:
            proc = subprocess.run(exec, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            proc_result = proc.stderr.decode('shift_jis')  # Possible inclusion of sjis-code
            match = re.search(regix, proc_result)
            if match is not None:
                group = match.group()
                result = int(group.split()[2])

        except subprocess.CalledProcessError:
            self.log.error('Failed to execute external program ({})'.format(cmd))

        return result

    def run(self):

        SYNC_WAIT_TIME = 2  # seconds: wait to stabilize sync before starting stream
        WAIT_TIME_SHORT = 1  # seconds: short wait for stabilization

        args = g_define.get_args()

        ctx = get_ctx()  # Access "SharedContext" set in parent process
        cv_frames = {}
        alive = True

        self.log.info('Thread \'FrameConsumer\' started.')

        # keyboard callback (Internal Functions)
        def keyboard_cb(key, scancode, action, mods):
            global g_value
            args = g_define.get_args()

            # Share values between external variables and within internal functions
            nonlocal ctx
            ctx = get_ctx()

            if (action == 1):  # Response to "onkeydown:action=1" and "onkeyup:action=0" being triggered twice.
                if key == UIKeyEvent.KEY_ESCAPE or key == UIKeyEvent.KEY_Q:
                    self.quit_mode = 1
                    self.log.info('OnKey Viewer End')

                elif key == UIKeyEvent.KEY_V:
                    if self.record_mode == 1 or self.as_seen_record_mode == 1:
                        return

                    self.view_mode = (self.view_mode + 1) % 4  # Switching between 0 and 3
                    msg = [k for k, v in self._VIEW_DICT.items() if v == self.view_mode][0]
                    self.log.info('OnKey View Mode: ' + msg)

                elif key == UIKeyEvent.KEY_R:
                    if self.as_seen_record_mode == 1:
                        return

                    self.record_mode = (self.record_mode + 1) % 2  # Switching between 0 and 1
                    if self.record_mode == 1:
                        ctx.params.recording_name.set('recording_' + datetime.now().strftime('%y%m%d_%H%M%S_%f')[:-3])
                        msg = 'On'

                        # Memo:
                        # Enable Invert to invert the polarity of the Output trigger signal.
                        # The Output trigger signal is separate from the event data of the target event camera (External Trigger).

                        self.cam.stop_streaming()
                        self.cam.LineSelector.set('Line0')
                        self.cam.LineSource.set('ExposureActive')

                        # Camera Frame Size getting
                        width_val = self.cam.Width.get()
                        height_val = self.cam.Height.get()

                        # Start Sync Recording
                        if args.cam_mode == 'master':
                            # Master
                            self.cam.LineSelector.set('Line2')
                            self.cam.LineInverter.set(False)  # False=ON, True=OFF

                        # Start event RAW
                        ctx.params.is_evt_recording.value = True

                        # Sync master waits here to avoid frame drops on the slave side.
                        if args.cam_mode == 'master':
                            time.sleep(SYNC_WAIT_TIME)

                        # Start frame MP4
                        self.log.info('create_mp4_process-start:' + datetime.now().strftime('%y%m%d_%H%M%S_%f')[:-3])
                        g_value.FFMPEG_PROCESS = createMp4Process(width_val, height_val)
                        self.log.info('create_mp4_process-end:' + datetime.now().strftime('%y%m%d_%H%M%S_%f')[:-3])
                        time.sleep(WAIT_TIME_SHORT)

                        g_value.is_recording = True
                        g_value.RECORDING_FRAME_CNT = 0

                        # Memo:
                        # The recording process in GPU mode rarely freezes.(ffmpeg)
                        # To prevent missing frames in such cases, increase the buffer size.
                        self.cam.start_streaming(FrameProducer(self.cam, self.frame_queue, self.event_queue), buffer_count=50)

                    else:
                        msg = 'Off'

                        if args.cam_mode == 'master':
                            # Master
                            self.cam.LineSelector.set('Line2')
                            self.cam.LineInverter.set(True)  # False=ON, True=OFF

                        # Sync slave waits here to avoid frame drops.
                        if args.cam_mode == 'slave':
                            time.sleep(SYNC_WAIT_TIME)

                        # Stop Sync Recording
                        self.cam.stop_streaming()

                        self.cam.LineSelector.set('Line0')
                        self.cam.LineSource.set('Off')
                        g_value.is_recording = False

                        destroyMp4Process(g_value.FFMPEG_PROCESS)

                        # Stop event RAW
                        time.sleep(WAIT_TIME_SHORT)
                        ctx.params.is_evt_recording.value = False

                        # Delay master stream start slightly to prevent the slave from recording unwanted frames.
                        if args.cam_mode == 'master':
                            time.sleep(SYNC_WAIT_TIME)

                        self.cam.start_streaming(FrameProducer(self.cam, self.frame_queue, self.event_queue))

                        file_path = os.path.join(args.recording_dir_path, ctx.params.recording_name.get())
                        thd = threading.Thread(target=self.execRecResultAnalysis, args=(file_path, args.output_result_trigger))
                        thd.start()

                    self.log.info('OnKey Record(2video) Mode: ' + msg)
                    self.log.info('During recording, the view is fixed to "frame mode"...')
                    self.view_mode = self._VIEW_DICT['frame']

                elif key == UIKeyEvent.KEY_M:
                    if self.record_mode == 1:
                        return

                    self.as_seen_record_mode = (self.as_seen_record_mode + 1) % 2  # Switching between 0 and 1

                    if self.as_seen_record_mode == 1:
                        ctx.params.recording_name.set('multi_recording_' + datetime.now().strftime('%y%m%d_%H%M%S_%f')[:-3])
                        msg = 'On'
                        g_value.is_as_seen_recording = True
                    else:
                        msg = 'Off'
                        g_value.is_as_seen_recording = False

                    self.log.info('OnKey Record(As seen) Mode: ' + msg)
                    self.view_mode = self._VIEW_DICT['multi']

        try:
            ctx = get_ctx()  # Reacquire shared context

            last_gc_time = time.time()
            while alive:
                # Dispatch system events to the window
                EventLoop.poll_and_dispatch()

                # Synchronous recording process between BothView (Slave side detection)
                if args.cam_mode == 'slave':
                    try:
                        event = self.event_queue.get(timeout=0)  # Instant confirmation of events

                        if event == "SYNC_RECORDING_ON" and not self.recording_on_processed:
                            if self.skip_processing:
                                # When the first “SYNC_RECORDING_ON” is detected, the “SYNC_RECORDING_OFF” side is unskipped
                                self.skip_processing = False

                            keyboard_cb(key=UIKeyEvent.KEY_R, action=1, scancode=None, mods=None)  # R ON
                            self.recording_on_processed = True
                            self.recording_off_processed = False  # Reset the other flag

                        elif event != "SYNC_RECORDING_ON" and not self.recording_off_processed and not self.skip_processing:
                            keyboard_cb(key=UIKeyEvent.KEY_R, action=1, scancode=None, mods=None)  # R OFF

                            self.recording_off_processed = True
                            self.recording_on_processed = False  # Reset the other flag

                    except queue.Empty:
                        pass  # If the queue is empty, nothing

                # Attempt to fetch one frame with timeout to avoid CPU spin
                if self.frame_queue.empty():
                    # CPU spin suppression with 1 ms sleep
                    time.sleep(0.001)
                    continue

                while not self.frame_queue.empty():
                    try:
                        cam_id, cv_frame = self.frame_queue.get_nowait()
                        cv_frames[cam_id] = cv_frame

                        # Perform garbage collection every 5 seconds based on real elapsed time
                        now = time.time()
                        if now - last_gc_time >= 5.0:
                            gc.collect()
                            last_gc_time = now

                    except queue.Empty:
                        break

                if not cv_frames:
                    continue

                # Coding based on the assumption that there is only one frame camera
                img_frame = cv_frames[next(iter(cv_frames))]
                img_event = ctx.image.get_image()

                if img_frame is None:
                    continue

                img_frame = img_frame[
                    g_value.img_trim_offset_y:(g_value.img_trim_offset_y + g_value.img_trim_height),
                    g_value.img_trim_offset_x:(g_value.img_trim_offset_x + g_value.img_trim_width)
                ]

                f_h, f_w, _ = numpy.shape(img_frame)
                img_event = cv2.resize(img_event, (f_w, f_h))
                view_base = self.getImageForViewMode(img_frame, img_event)
                self.selectRecordMode(view_base)

                if g_value.is_recording:
                    mode = 'rec'
                    description = '[REC(2video)]'
                    color = (0, 0, 255)  # R

                elif g_value.is_as_seen_recording:
                    mode = 'as_seen'
                    description = '[REC(multi)]'
                    color = (0, 0, 255)  # R

                elif g_value.is_rec_result_proc:
                    mode = 'analyze'
                    description = 'Analyzing results. Please wait...'
                    color = (0, 255, 0)  # G
                    self.record_mode = 0

                else:
                    mode = None

                # Regenerate overlay only when state changes
                if mode != self.overlay_mode:
                    self.overlay = None
                    self.overlay_mode = mode

                if mode:
                    if self.overlay is None:
                        self.overlay = numpy.zeros_like(view_base)
                        cv2.putText(self.overlay, description, (30, 30),  # outline color
                                    cv2.FONT_HERSHEY_COMPLEX_SMALL, 1, (30, 30, 30), 2, cv2.LINE_AA)
                        cv2.putText(self.overlay, description, (30, 30),  # main color
                                    cv2.FONT_HERSHEY_COMPLEX_SMALL, 1, color, 1, cv2.LINE_AA)

                    view_overlay = cv2.addWeighted(view_base, 1.0, self.overlay, 1.0, 0)

                else:
                    view_overlay = view_base

                if self.window is None:
                    title_plus = ''
                    title_plus = args.event_camera_serial
                    if args.cam_mode == 'master':
                        title_plus += ' [Master]'
                    elif args.cam_mode == 'slave':
                        title_plus += ' [Slave]'

                    # Initial window display size
                    self.window = MTWindow(
                        title=self._IMAGE_CAPTION + ' ' + title_plus,
                        width=f_w,
                        height=f_h,
                        mode=BaseWindow.RenderMode.BGR,
                        open_directly=True
                    )
                    self.window.set_keyboard_callback(keyboard_cb)

                if not self.window.should_close() and self.quit_mode == 0:
                    self.window.show_async(view_overlay)
                    del view_overlay
                    del view_base
                else:
                    if g_value.is_recording:
                        self.log.info("It was being recorded(2video). Recording will be terminated...")
                        keyboard_cb(key=UIKeyEvent.KEY_R, action=1, scancode=None, mods=None)
                        time.sleep(WAIT_TIME_SHORT)
                    elif g_value.is_as_seen_recording:
                        self.log.info("It was being recorded(multi). Recording will be terminated...")
                        keyboard_cb(key=UIKeyEvent.KEY_M, action=1, scancode=None, mods=None)
                        time.sleep(WAIT_TIME_SHORT)
                    break

            del cv_frames

        finally:
            self.closeCamera()
            self.log.info("Thread 'FrameConsumer' terminated.")

    def selectRecordMode(self, img_frame: numpy.ndarray):

        # Select record mode
        if g_value.is_as_seen_recording:
            if not self.init_recording:

                # 1.Start Recording
                self.log.info('create_mp4_process-start:' + datetime.now().strftime('%y%m%d_%H%M%S_%f')[:-3])
                img_h, img_w, _ = numpy.shape(img_frame)

                # Specifies the bit rate value at which 'block noise' is not generated.
                self.ffmpeg_process = createMp4Process(img_w, img_h)
                self.log.info('create_mp4_process-end:' + datetime.now().strftime('%y%m%d_%H%M%S_%f')[:-3])

                self.init_recording = True
            else:
                try:
                    self.ffmpeg_process.stdin.write(img_frame)
                except Exception as e:
                    # Select 'GPU' and follow up if GPU is not available.
                    self.log.error('ffmpeg error:' + str(e))
                    g_value.is_as_seen_recording = False
                    self.as_seen_record_mode = 0
                    self.init_recording = False

        else:
            if self.init_recording:

                try:
                    # Save the last frame
                    self.ffmpeg_process.stdin.write(img_frame)
                except Exception as e:
                    # Select 'GPU' and follow up if GPU is not available.
                    self.log.error('ffmpeg error:' + str(e))

                # 2.End of Recording
                destroyMp4Process(self.ffmpeg_process)

            self.init_recording = False

    def getImageForViewMode(self, img_frame: numpy.ndarray, img_event: numpy.ndarray):

        # Composite image (Internal Functions)
        def doImageComposition(img_frame: numpy.ndarray, img_event: numpy.ndarray):
            # alpha and beta are added together, and the larger number is used rather than 1.0 because it was easier to see.
            return cv2.addWeighted(
                src1=img_frame, alpha=0.8,
                src2=img_event, beta=0.8,
                gamma=0
            )

        # Create split image (Internal Functions)
        def doImageConcatenation(im_list_2d: list):
            return cv2.vconcat([cv2.hconcat(im_list_h) for im_list_h in im_list_2d])

        if self.view_mode == self._VIEW_DICT['both']:
            img_display = doImageComposition(img_frame, img_event)
        elif self.view_mode == self._VIEW_DICT['event']:
            img_display = img_event
        elif self.view_mode == self._VIEW_DICT['frame']:
            img_display = img_frame
        elif self.view_mode == self._VIEW_DICT['multi']:
            img_display = doImageComposition(img_frame, img_event)
            img_display_s = cv2.resize(img_display, dsize=(0, 0), fx=0.5, fy=0.5)
            img_event_s = cv2.resize(img_event, dsize=(0, 0), fx=0.5, fy=0.5)
            img_frame_s = cv2.resize(img_frame, dsize=(0, 0), fx=0.5, fy=0.5)

            if not self.init_multi_view:
                re_h, re_w, _ = numpy.shape(img_display_s)
                self.desc_list[0] = cv2.resize(self.desc_list[0], dsize=(re_w, re_h))
                self.desc_list[1] = cv2.resize(self.desc_list[1], dsize=(re_w, re_h))
                self.init_multi_view = True

            dt = datetime.now()
            sec_2nd_digit = math.floor(dt.second / 10)
            img_description = self.desc_list[0] if sec_2nd_digit % 2 == 0 else self.desc_list[1]

            img_display = doImageConcatenation([[img_display_s, img_frame_s], [img_event_s, img_description]])
            del img_display_s
            del img_event_s
            del img_frame_s
            del img_description

            del img_frame
            del img_event

        return img_display


class MainThread(threading.Thread):
    def __init__(self):
        super().__init__()

        self.frame_queue = queue.Queue(maxsize=FRAME_QUEUE_SIZE)
        self.event_queue = queue.Queue(maxsize=1)  # Receive “event_queue” from FrameConsumer
        self.producers = {}
        self.producers_lock = threading.Lock()  # Creating Lock Objects

    def __call__(self, cam: Camera, event: CameraEvent):
        if event == CameraEvent.Detected:
            with self.producers_lock:
                self.producers[cam.get_id()] = FrameProducer(cam, self.frame_queue, self.event_queue)
                self.producers[cam.get_id()].start()

        elif event == CameraEvent.Missing:
            with self.producers_lock:
                producer = self.producers.pop(cam.get_id())
                producer.stop()
                producer.join()

    def run(self):
        args = g_define.get_args()

        log = Log.get_instance()
        ctx = get_ctx()  # Access "SharedContext" set in parent process

        consumer = None
        vmb = VmbSystem.get_instance()
        vmb.enable_log(LOG_CONFIG_INFO_CONSOLE_ONLY)

        log.info('Thread \'MainThread\' started.')

        with vmb:
            cam = None
            try:
                cams = vmb.get_all_cameras()
                cam = cams[0]

                if args.frame_camera_serial != '':
                    target_flag = False
                    for c in cams:
                        if c.get_serial() == args.frame_camera_serial:
                            target_flag = True
                            cam = c
                            break
                    if not target_flag:
                        raise IndexError("No frame camera for the specified serial value of the parameter.")
            except (IndexError):
                log.error('[Error] Frame camera not found. Check the USB connection.')
                ctx.params.frame_camera_err_flag.set()

            ctx.params.frame_camera_open_proc.set()
            if ctx.params.frame_camera_err_flag.is_set():
                return

            # Wait for the end of the 'event camera' opening
            ctx.params.event_camera_open_proc.wait()
            if ctx.params.event_camera_open_proc.is_set():
                if ctx.params.event_camera_err_flag.is_set():
                    return  # unavoidably terminated

            self.producers[cam.get_id()] = FrameProducer(cam, self.frame_queue, self.event_queue)
            consumer = FrameConsumer(cam, self.frame_queue, self.event_queue)

            with self.producers_lock:
                for producer in self.producers.values():
                    producer.start()

            # Start and wait for consumer to terminate
            vmb.register_camera_change_handler(self)
            consumer.start()
            consumer.join()
            vmb.unregister_camera_change_handler(self)

            # Stop all FrameProducer threads
            with self.producers_lock:
                # Initiate concurrent shutdown
                for producer in self.producers.values():
                    producer.stop()

                # Wait for shutdown to complete
                for producer in self.producers.values():
                    producer.join()

        log.info('Thread \'MainThread\' terminated.')


def run_main_thread(shm_name: str, shm_lock: LockType, shared_params: SharedParams):

    # Initialize “ctx” in child process
    init_ctx(name=shm_name, create=False, lock=shm_lock, shared_params=shared_params)

    main = MainThread()
    main.start()
    main.join()
    gc.collect()


if __name__ == '__main__':
    # Configuration for "Windows" and "Ubuntu" compatibility
    multiprocessing.set_start_method("spawn")

    # For help display
    g_define.parse_args()
    print_preamble()

    # Create unique memory name with timestamp
    ts = time.strftime("%Y%m%d_%H%M%S")
    shm_name = f'my_shm_{ts}'
    shm_lock = multiprocessing.Lock()
    manager = multiprocessing.Manager()
    shared_params = SharedParams(manager)
    init_ctx(name=shm_name, create=True, lock=shm_lock, shared_params=shared_params)
    ctx = get_ctx()

    # Process initialization
    frame_process = multiprocessing.Process(target=run_main_thread, args=(shm_name, shm_lock, shared_params))
    event_process = multiprocessing.Process(target=silky_cam.main, args=(shm_name, shm_lock, shared_params))

    try:
        # Thread Start
        frame_process.start()
        event_process.start()

        # Thread Wait
        frame_process.join()
        event_process.join()

    finally:
        ctx.close()
        ctx.unlink()
