# Copyright (c) 2025 CenturyArks Co.,Ltd.
#
# These source code may be used (including the use of modified source code) only when using
# CenturyArks's products or services.
#
# Although we have confirmed that it works in the copyright holder's environment, we cannot guarantee that
# there will be no defects.
#
# If you modify the source code, do not remove the copyright notice.
#
# Copyright holder shall not be liable for any claim, damages or other liability arising
# from or relating to the source code or arising from the use or other dealings in the software.

"""
'2Video Sync Recording' Result File Output(Event Raw Data)
"""
import argparse
import os
import re
from pathlib import Path

from bothview_main import FrameConsumer
from vmbpy import LOG_CONFIG_INFO_CONSOLE_ONLY, Log

RECORDING_DIR = os.path.join(os.path.dirname(os.path.abspath(__file__)), "video")


def parse_args():
    """Parse command-line arguments."""
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "-i", "--input",
        dest="target_raw_file",
        default=None,
        help="Specify the target raw file in the format 'recording_YYMMDD_HHMMSS_SSS.raw'."
    )

    default_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), "video"))
    parser.add_argument(
        "-dir", "--recording-dir-path",
        dest="recording_dir_path",
        default=default_dir,
        type=str,
        help="Specify the directory for saving recorded files (default: './video' relative to the script location)."
    )
    return parser.parse_args()

def find_latest_file_pattern(directory: str = RECORDING_DIR):
    log = Log.get_instance()
    log.enable(LOG_CONFIG_INFO_CONSOLE_ONLY)

    pattern = re.compile(r'^recording_(\d{6}_\d{6}_\d+)\.raw$')
    candidates = []

    for filename in os.listdir(directory):
        match = pattern.match(filename)
        if match:
            key = match.group(1)
            candidates.append((key, filename))

    if not candidates:
        log.error(f"[Error] The target recording_YYMMDD_HHMMSS_SSS.raw file was not found in '{directory}'")
        return None

    latest = max(candidates, key=lambda x: x[0])
    log.info(f'Latest file pattern:{latest[1]}')
    return latest[1]  # Return with extension


def update_or_prepend_line(file_path: str, ext_trigger_num: int, line_identifier: str = "Event Result Triggers: ") -> None:
    """
    Do the following to the specified file:
    - If the file does not exist, create it and write new_line
    - If the file contains a line with line_identifier, overwrite it with new_line
    - If not, add new_line to the beginning of the file

    Args:
    file_path (str): Path to the target file
    ext_trigger_num (int): Number of external triggers
    line_identifier (str): String identifying the target line (default: "Event Ext Triggers: ")
    """
    path = Path(file_path)
    write_line = line_identifier + str(ext_trigger_num) + '\n'

    if not path.exists():
        # If the file does not exist, create a new one
        with open(path, 'w') as f:
            f.write(write_line)
        return

    # Load the file if it exists
    with open(path, 'r') as f:
        lines = f.readlines()

    # Replace or prepend the line
    replaced = False
    for i, line in enumerate(lines):
        if line.startswith(line_identifier):
            lines[i] = write_line
            replaced = True
            break

    if not replaced:
        # If the file has no lines, insert at index 0
        insert_index = 1 if len(lines) >= 1 else 0
        lines.insert(insert_index, write_line)

    with open(path, 'w') as f:
        f.writelines(lines)


def main():
    log = Log.get_instance()
    log.enable(LOG_CONFIG_INFO_CONSOLE_ONLY)

    args = parse_args()

    if args.target_raw_file:
        raw_file = args.target_raw_file
        log.info(f"Use the file specified in the startup parameter: {raw_file}")
    else:
        raw_file = find_latest_file_pattern(directory=args.recording_dir_path)
        if raw_file is None:
            log.error("[Error] Processing will be stopped")
            return

    raw_file_path = os.path.join(args.recording_dir_path, raw_file)
    if not Path(raw_file_path).exists():
        log.error(f"[Error] The specified file was not found: {raw_file_path}")
        return

    pattern_path = os.path.splitext(raw_file_path)[0]  # Make the path without the extension
    result_file_path = pattern_path + '_result.txt'

    log.info(f"Start analyzing file... '{raw_file_path}'")
    consumer = FrameConsumer()
    ext_trigger_num = consumer.getResultEventNum(raw_file_path)
    log.info(f"Number of external triggers taken: {ext_trigger_num}")
    update_or_prepend_line(result_file_path, ext_trigger_num)
    log.info(f"Finish analyzing file... The text of the recording results is output to '{result_file_path}'")


if __name__ == "__main__":
    main()
