# Copyright (c) 2025 CenturyArks Co.,Ltd.
#
# These source code may be used (including the use of modified source code) only when using
# CenturyArks's products or services.
#
# Although we have confirmed that it works in the copyright holder's environment, we cannot guarantee that
# there will be no defects.
#
# If you modify the source code, do not remove the copyright notice.
#
# Copyright holder shall not be liable for any claim, damages or other liability arising
# from or relating to the source code or arising from the use or other dealings in the software.

"""
Parameter Defines
"""
import argparse
import os


def parse_args():
    """Parse command line arguments."""
    parser = argparse.ArgumentParser(description='CenturyArks BothView sample.',
                                     formatter_class=argparse.ArgumentDefaultsHelpFormatter)

    # Memo:Event H/W(カメラ設定 に関連)
    # イベントカメラのシリアルナンバー。未指定の場合、最初に使用可能なイベントカメラのライブ ストリームが使用されます。(例:00000001)
    input_group = parser.add_argument_group('Input', 'Arguments related to input sequence.')
    input_group.add_argument(
        '-se', '--event-camera-serial', dest='event_camera_serial', default='',
        help='Serial number of the event camera. If not specified, the live stream of the first available event camera will be used.')

    # Memo:RGB H/W(カメラ設定 に関連)
    # フレームカメラのシリアルナンバー。未指定の場合、最初に使用可能なフレームカメラのライブ ストリームが使用されます。(例:01AAA)
    input_group.add_argument(
        '-sf', '--frame-camera-serial', dest='frame_camera_serial', default='',
        help='Serial number of the frame camera. If not specified, the live stream of the first available frame camera will be used.')

    # Memo:Event S/W(ビューア表示 に関連)
    # 蓄積時間。フレームの生成に使用されます。(単位:us)
    input_group.add_argument(
        '-at', '--accumulation-time-us', dest='accumulation_time_us', default=33333, type=ranged_type(int, 1, 1000000),
        help='Accumulation time. Used to generate a frame. (Unit:us)')

    # Memo:RGB H/W(カメラ設定 に関連)
    # フレームカメラからの取得FrameRate設定値を有効にします。(無効の場合、FrameRate値は露光時間を基準とします)
    # ※「(撮影環境が明るい等の理由で、)露光時間を小さく設定した場合でも、FrameRate値を一定にする」目的で使用します。
    # ※取得FrameRate設定値の1フレーム相当の時間が露光時間より短い場合(例: 1frame=100ms[10fps], 露光時間=200ms[5fps])は、
    # 　露光時間を基準とし、本設定は無効となります。(その際、コンソールログに出力されます)
    input_group.add_argument(
        '-eaf-mode', '--enable-acquisition-frame-mode', dest='eaf_mode', action='store_true',
        help='Enables the FrameRate specified value for the frame camera. (If disabled, the FrameRate value is based on the exposure time.)')

    # Memo:RGB H/W(カメラ設定 に関連)
    # フレームカメラからの取得FrameRate設定値。FrameRate値を制御します。(単位:fps)
    input_group.add_argument(
        '-a-frame', '--acquisition-frame', dest="acquisition_frame", default=15, type=ranged_type(float, 1.0, 30.0),
        help='FrameRate specified value for frame camera. controls the FrameRate value. (Unit: fps)')

    # Memo:Event H/W(カメラ設定 に関連)
    # Bias設定ファイル(./config/biases.bias)を有効にします。その際、イベントカメラ設定ファイル(./config/settings.json)のBIAS値を上書きします。
    input_group.add_argument(
        '-ub', '--using-bias-file', dest='using_bias_file', action='store_true',
        help="Enable the Bias configuration file (. /config/biases.bias) is enabled. In doing so, "
             "the BIAS values in the event camera configuration file (. /config/settings.json).")

    # Memo:Event,RGB H/W(カメラ設定 に関連)
    # BothView カメラに対して選択された複数台同期モード: マスターまたはスレーブ。指定しない場合、カメラは単一 モードになります。
    input_group.add_argument(
        '-m', '--mode-selection', dest='cam_mode', default='',
        help="Multi-device synchronization mode selected for BothView cameras: Master or Slave. If not specified, the camera will be in single mode.")

    # Memo:RGB H/W(カメラ設定 に関連)
    # 一部の Allied Vision U3V カメラで、XML 設定適用後にストリーミングが正常に開始されない事例があるため、それを回避するためのワークアラウンドです。
    # XML 読み込みをスキップし、UserSet の値のみを使用します。
    input_group.add_argument(
        '-xml0', '--skip-xml-load', dest='skip_xml_load', action='store_true',
        help=(
            "Skip XML loading and use UserSet values only. Included as a workaround to avoid a "
            "condition observed with some Allied Vision U3V cameras where streaming may not "
            "start properly after applying XML settings."
        )
    )

    # Memo:Event S/W(ビューア表示 に関連)
    # サンプルコード上でイベントカメラ画像の反転処理を無効にします。(ビームスプリッターレンズを使用しているため）
    image_setting_group = parser.add_argument_group('Image Setting', 'Arguments related to image display.')
    image_setting_group.add_argument(
        '-dr', '--disable-reverse', dest='disable_reverse', action='store_true',
        help='Disable the event camera image inversion process on the sample code. (because of the use of a beam splitter lens)')

    # Memo:RGB S/W(ビューア表示 / MP4録画ファイル に関連)
    # 'config.ini'ファイルで指定しているフレームカメラの調整値(adjust_view_w, adjust_view_h)を無効にします。
    image_setting_group.add_argument(
        '-da', '--disable-adjust', dest='disable_adjust', action='store_true',
        help="Disables the frame camera adjustment values (adjust_view_w, adjust_view_h) specified in the 'config.ini' file.")

    # Memo:Event S/W(ビューア表示 に関連)
    # ビューアで表示するフレームレートを指定します。(カメラのフレーム出力から均等に間引き。割り切れない際は近似値)
    # カメラのフレームレート(Acquisition Frame Rate)値の方が本値より小さい場合、カメラのフレームレート値がビューアフレームレート値となります。
    # ※録画ファイルでのフレームレートとは異なります。
    image_setting_group = parser.add_argument_group('Image Setting', 'Arguments related to image display.')
    image_setting_group.add_argument(
        '-fr', '--viewer-frame-rate', dest='viewer_frame_rate', default=20, type=ranged_type(int, 1, 30),
        help='Specifies the frame rate to be displayed in the viewer. Thinned out evenly from the camera''s frame output. '
             'Approximate value if not divisible. (Unit:fps)')

    # Memo:RGB S/W(MP4録画ファイル に関連)
    # フレームカメラ動画の録画処理に'NVIDIA GPU'を使用します。(指定がない場合はCPUが使用されます)
    # * 'NVIDIA GPU'を使用する事を推奨しますが、実行PCの環境設定が別途必要です。(GPUドライバ、cudaSDK)
    recording_setting_group = parser.add_argument_group('Recording Setting', 'Arguments related to recording.')
    recording_setting_group.add_argument(
        '-ug', '--frame-using-gpu', dest='frame_using_gpu', action="store_true",
        help='Use \'NVIDIA GPU\' for frame camera video recording process.(If not specified, CPU is used)')

    # Memo:RGB S/W(MP4録画ファイル に関連)
    # フレームカメラから割り振られたフレームIDを録画画像に書き出します。
    # ※フレームが欠落する(カメラデータ出力時、mp4書き出し時)場合があり、その際どのフレームが欠落したかを確認するのに有効です。
    recording_setting_group.add_argument(
        '-wf', '--write-frame-id', dest='write_frame_id', action='store_true',
        help='The frame ID assigned from the frame camera is written to the recorded image.')

    # Memo:Event & RGB S/W(録画機能 に関連)
    # 録画機能の最後で出力される、実行結果ファイル内、イベントカメラのRawデータ 外部トリガー数の出力を有効にします。
    # ※イベントカメラのRawデータ 外部トリガー数の出力には(録画時間、イベント数に比例して)時間がかかる為、デフォルトは無効としています。
    recording_setting_group.add_argument(
        '-ort', '--output-result-trigger', dest='output_result_trigger', action='store_true',
        help="Enables output of the raw data external trigger count of the event camera in the execution result file"
             "that is output at the end of the 'recording function'.")

    # Memo:Event & RGB S/W(録画機能 に関連)
    # 録画ファイルの保存先ディレクトリを指定します（デフォルトはサンプルコードの実行ディレクトリを基点とした "./video"）。
    default_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), "video"))
    recording_setting_group.add_argument(
        '-dir', '--recording-dir-path', dest='recording_dir_path', default=default_dir, type=str,
        help="Specify the directory for saving recorded files (default: './video' relative to the sample code location).")

    args = parser.parse_args()

    return args


def ranged_type(value_type, min_value, max_value):
    def range_checker(arg: str):
        try:
            f = value_type(arg)
        except ValueError:
            raise argparse.ArgumentTypeError(f'must be a valid {value_type}')
        if f < min_value or f > max_value:
            raise argparse.ArgumentTypeError(f'must be within [{min_value}, {max_value}]')
        return f

    return range_checker


# Global cache variable (default value: None)
_args_cache = None


def get_args():
    # A function that caches and returns the result of 'parse_args()'
    global _args_cache
    if _args_cache is None:
        _args_cache = parse_args()
    return _args_cache
