# Copyright (c) 2025 CenturyArks Co.,Ltd.
#
# These source code may be used (including the use of modified source code) only when using
# CenturyArks's products or services.
#
# Although we have confirmed that it works in the copyright holder's environment, we cannot guarantee that
# there will be no defects.
#
# If you modify the source code, do not remove the copyright notice.
#
# Copyright holder shall not be liable for any claim, damages or other liability arising
# from or relating to the source code or arising from the use or other dealings in the software.

import json
import traceback
from metavision_hal import I_EventTrailFilterModule, I_AntiFlickerModule
from vmbpy import Log


class CameraConfig():
    """
    Event Camera Setting file read and set
    """

    dict_json = None

    def __init__(self, device):
        self.log = Log.get_instance()
        self.device = device

    def __snake_to_camel(self, snake_str):
        components = snake_str.split('_')
        return ''.join(x.title() for x in components)

    def __erc_conversion(self, device_erc, cd_event_count):
        calc_value = 1e6 / device_erc.get_count_period()  # 1e6(microsec) = 1sec
        return int(cd_event_count * calc_value)

    def set_from_file(self, file_name='settings.json', disp_flag=True):

        try:
            f = open(file_name, 'r')
            self.dict_json = json.load(f)

            serial = self.device.get_i_hw_identification().get_system_info()['Serial']
            self.log.info('--> [CameraConfig] Event Camera has been opened ({})'.format(serial))

            # Set Item 1. Anti Flicker
            if self.device.get_i_antiflicker_module():
                # we test if the facility is available on this device before using it
                device_antiflicker = self.device.get_i_antiflicker_module()
                json_antiflicker = self.dict_json['afk_state']

                device_antiflicker.set_frequency_band(
                    min_freq=json_antiflicker['band_low_freq'],
                    max_freq=json_antiflicker['band_high_freq']
                )
                device_antiflicker.set_start_threshold(json_antiflicker['start_threshold'])
                device_antiflicker.set_stop_threshold(json_antiflicker['stop_threshold'])
                filtering_mode = self.__snake_to_camel(json_antiflicker['filtering_mode'])
                device_antiflicker.set_filtering_mode(
                    getattr(I_AntiFlickerModule.AntiFlickerMode, filtering_mode)
                )
                device_antiflicker.set_duty_cycle(json_antiflicker['duty_cycle'])
                device_antiflicker.enable(json_antiflicker['enabled'])

                if disp_flag:
                    self.log.info('-->[CameraConfig] Set "Anti Flicker":{}'.format(json_antiflicker))

            # Set Item 2. Event Trail Filter
            if self.device.get_i_event_trail_filter_module():
                # we test if the facility is available on this device before using it
                device_eventtrailfilter = self.device.get_i_event_trail_filter_module()
                json_eventtrailfilter = self.dict_json['event_trail_filter_state']

                device_eventtrailfilter.set_type(
                    getattr(I_EventTrailFilterModule.Type, json_eventtrailfilter['filtering_type'])
                )
                device_eventtrailfilter.set_threshold(json_eventtrailfilter['threshold'])
                device_eventtrailfilter.enable(json_eventtrailfilter['enabled'])

                if disp_flag:
                    self.log.info('-->[CameraConfig] Set "Event Trail Filter":{}'.format(json_eventtrailfilter))

            # Set Item 3. Biases
            if self.device.get_i_ll_biases():
                # we test if the facility is available on this device before using it
                device_biases = self.device.get_i_ll_biases()
                json_biases = self.dict_json['ll_biases_state']['bias']
                bias_values = {item["name"]: item["value"] for item in json_biases}

                device_biases.set("bias_diff", bias_values["bias_diff"])
                device_biases.set("bias_diff_off", bias_values["bias_diff_off"])
                device_biases.set("bias_diff_on", bias_values["bias_diff_on"])
                device_biases.set("bias_fo", bias_values["bias_fo"])
                device_biases.set("bias_hpf", bias_values["bias_hpf"])
                device_biases.set("bias_refr", bias_values["bias_refr"])

                if disp_flag:
                    self.log.info('-->[CameraConfig] Set "Biases":{}'.format(bias_values))

            # Set Item 4. Event Rate Control
            if self.device.get_i_erc_module():
                # we test if the facility is available on this device before using it
                device_erc = self.device.get_i_erc_module()
                json_erc = self.dict_json['event_rate_control_state']

                events_per_sec = self.__erc_conversion(device_erc, json_erc['cd_event_count'])
                device_erc.set_cd_event_rate(events_per_sec)
                device_erc.enable(json_erc['enabled'])

                if disp_flag:
                    self.log.info('-->[CameraConfig] Set "Event Rate Control":{}'.format(json_erc))

                '''
                # For debugging
                print('event-rate-count: ' + str(device_erc.get_cd_event_count()))
                print('event-rate: ' + str(device_erc.get_cd_event_rate()))
                print('MAX-event-rate-count: ' + str(device_erc.get_max_supported_cd_event_count()))
                print('MAX-event-rate: ' + str(device_erc.get_max_supported_cd_event_rate()))
                print('MIN-event-rate-count: ' + str(device_erc.get_min_supported_cd_event_count()))
                print('MIN-event-rate: ' + str(device_erc.get_min_supported_cd_event_rate()))
                '''

                self.log.info('--> [CameraConfig] Feature values have been loaded from given file \'%s\'' % file_name)

        except Exception:
            traceback.print_exc()

        finally:
            f.close()


class BiasConfig():
    """
    Bias file read and set
    """

    def __init__(self, biases):
        self.log = Log.get_instance()
        self.biases = biases

    def set_from_file(self, file_name='biases.bias', disp_flag=True):
        try:
            f = open(file_name, 'r', encoding='UTF-8')
            datalist = f.readlines()
            for data in datalist:
                if len(data) <= 1:
                    # Only new line.
                    continue
                elif data.strip()[0] == '#':
                    # Leading # is a comment.
                    continue

                data2 = data.split('%')
                if len(data2) != 2:
                    continue

                v1 = int(data2[0].strip())
                s1 = data2[1].strip()
                self.biases.set(s1, v1)

        except Exception:
            traceback.print_exc()

        if disp_flag:
            dict_biases = self.biases.get_all_biases()
            self.log.info('-->[BiasConfig] Current "Biases":{}'.format(dict_biases))
